#!/bin/sh
#
# This file is part of the FreeWRT project. FreeWRT is copyrighted
# material, please see the LICENCE file in the top-level directory
# or at http://www.freewrt.org/licence for details.
#
# Christian Fischer <spaetzle@freewrt.org>
#

set -u

builtin() {
	local builtin=$1
	local alias=$(alias $builtin)
	unalias $builtin
	alias $(echo $alias|sed 's/'\''//g')
	shift
	$builtin $@
}

config() {
	[ "$1" != "0" ]
}
__ifupdown() {
	local environ=$(cat $ENVFILE)

	IFUPDOWN_ENV=$(echo -n $IFUPDOWN_ENV|sed 's/\(MDENT=\)[^ ]*[[:space:]]*//g';\
		echo " MDENT=$MDENT"; echo -e " MOFFSET=$MOFFSET") busybox $@

	local retval=$?
	echo $environ > $ENVFILE
	return $retval
}

__exit() {
	[ "$1" != "0" ] && echo "H_ERR=1" >> $ENVFILE
	exit $1
}

iface_exists() {
	grep -q "${1:-$IFACE}:" /proc/net/dev
}

is_up() {
	local iface=${1:-$IFACE}

	if iface_exists $iface
	then
		ip a|grep ${iface}:[[:space:]]|grep -q UP && return

		### ifupdown netlink bug workaround ###
		# suppress "ip: RTNETLINK answers: File exists" message in the case of
		# iface is down but an ip address is assigned
		ip a|grep ${iface}$|grep -q inet && ip addr flush dev $iface >/dev/null 2>&1
		### end workaround ###
	fi
	return 1
}

env_update() {
	local lastmatch=""

	# get submode from calling hook script
	SUBMODE=$(dirname $0|sed '{ s/\(.*\/\)*//; s/\.d$//}')

	# create env file
	if ! [ -f "$ENVFILE" ]
	then
		[ -d $(dirname $ENVFILE) ] || mkdir -p $(dirname $ENVFILE)
		echo "MDENT=0" >> $ENVFILE
		echo "LASTLOG=0" >> $ENVFILE

		# search for existing lib files end evaluate iface type by using the appropriate
		# function from lib file
		for lib in ${LIBDIR}/* ${LIBDIR_OVERLAY}/*
		do
			if [ -f $lib ]
			then
				if (. $lib; iface_type 2>&-)
				then
					if [ "$lastmatch" != "" ]
					then
						if [ "$(basename $lib)" = "$(basename $lastmatch)" ]
						then
							logger "Warning: $lib overlays $lastmatch"
						else
							logger "Error: iface type $lib overlays $lastmatch"
						fi
					fi
					lastmatch=$lib
				fi
			fi
		done

		if [ "$lastmatch" != "" ]
		then
			IFACE_TYPE=$(basename $lastmatch)
		else
			# if iface type isn't evaluated type "iface" is default
			IFACE_TYPE="iface"
		fi

		# update env
		echo "IFACE_TYPE=${IFACE_TYPE}" >> $ENVFILE
	fi

	. $ENVFILE
	# expand $IFUPDOWN_ENV
	for var in $IFUPDOWN_ENV; do
		eval $var
	done
	pos_update
}

pos_update() {
	if [ "${MDENT:-0}" = "0" ]
	then
		STARTCOL=""
	#	MPREFIX=$MP_GOOD
	else
		STARTCOL="\x1B[$(expr ${MDENT:-0} \* ${TAB})C"
	#	MPREFIX="\x1B[3C"
	fi
	sed 's/MDENT=[0-9]*/'MDENT=$MDENT'/' -i $ENVFILE
}




### main entry point ###

CFG_PRINTING_OFF=0
CFG_SYSLOG_OFF=0
CFG_BUSYBOX_COMPAT=0
CFG_START_PRINTING_OFF=0
CFG_STOP_PRINTING_OFF=0
CFG_PREUP_IFUP_CHECK_OFF=0
CFG_UP_IFUP_CHECK_OFF=0
CFG_DOWN_IFUP_CHECK_OFF=0
CFG_POSTDOWN_IFUP_CHECK_OFF=0

if [ -f /etc/conf.d/ifupdown ]; then
	. /etc/conf.d/ifupdown
	[ "$CFG_BUSYBOX_COMPAT" = "1" ] && exit 1
else
	exit 1
fi

ENVFILE="/tmp/ifupdown/env"
LIBDIR="/opt/ifupdown/lib"
LIBDIR_OVERLAY="/etc/network/lib"
IFACE_TYPE="none"

alias exit="__exit"
alias ifup="__ifupdown ifup"
alias ifdown="__ifupdown ifdown"

. /opt/ifupdown/bin/mfunctions.sh

env_update

if [ "IFACE_TYPE" != "iface" ]
then
	if [ -f "${LIBDIR_OVERLAY}/$IFACE_TYPE" ]
	then
		. ${LIBDIR_OVERLAY}/$IFACE_TYPE
	elif [ -f "${LIBDIR}/$IFACE_TYPE" ]
	then
		. ${LIBDIR}/$IFACE_TYPE
	else
		logger "Error file not found and abort"
	fi
fi

[ "${H_ERR:=""}" = "1" ] && builtin exit 1

case $SUBMODE in
	if-pre-up)
		config $CFG_START_PRINTING_OFF || mstart

		if ! config $CFG_PREUP_IFUP_CHECK_OFF && is_up
		then
			if ! config $CFG_START_PRINTING_OFF
			then
				mup
				mstate 1
				minfo "Interface already up, run ifdown ${IFACE} first."
			fi
			exit 1
		fi

		if_preup 2>&-
		retval=$?
		if [ "$retval" != "0" ]
		then
			[ $retval != "127" ] && exit 1
		fi

		if ! config $CFG_START_PRINTING_OFF
		then
			if [ "$METHOD" = "manual" ]
			then
				minfo "Can not bringing up ${IFACE}, method manual is set"
			else
				mup
			fi
		fi
	;;
	if-up)
		if ! config $CFG_UP_IFUP_CHECK_OFF
		then
			if ! config $CFG_START_PRINTING_OFF
			then
				is_up
				mstate $?
			fi
		fi

		if_up 2>&-
		retval=$?
		if [ "$retval" != "0" ]
		then
			[ $retval != "127" ] && exit 1
		fi

	;;
	if-down)
		config $CFG_STOP_PRINTING_OFF || mstop

		if ! config $CFG_DOWN_IFUP_CHECK_OFF && ! is_up
		then
			if ! config $CFG_STOP_PRINTING_OFF
			then
				mdown
				mstate 1
				minfo "Can not bringing down, interface $IFACE is down."
			fi
			exit 1
		fi

		if_down 2>&-
		retval=$?
		if [ "$retval" != "0" ]
		then
			[ $retval != "127" ] && exit 1
		fi

		if ! config $CFG_STOP_PRINTING_OFF
		then
			if [ "$METHOD" = "manual" ]
			then
				minfo "Can not bringing down ${IFACE}, method manual is set"
			else
				mdown
			fi
		fi
	;;
	if-post-down)
		if ! config $CFG_POSTDOWN_IFUP_CHECK_OFF
		then
			if ! config $CFG_STOP_PRINTING_OFF
			then
				! is_up
				mstate $?
			fi
		fi

		if_postdown 2>&-
		retval=$?
		if [ "$retval" != "0" ]
		then
			[ $retval != "127" ] && exit 1
		fi

	;;
esac

# vim:ts=4
